/**
運用のワンポイント（任意）
切断が多い環境なら：SIO_PING_INTERVAL=15000 / SIO_PING_TIMEOUT=45000 に短縮すると安定することがあります。
Nginx等のリバプロを挟む場合：
proxy_http_version 1.1;、proxy_set_header Upgrade $http_upgrade;、proxy_set_header Connection "upgrade";
タイムアウトは proxy_read_timeout 75s 以上 を目安に。
クラスター運用（PM2 clusterや複数インスタンス）なら Socket.IO はスティッキーセッションか Redis Adapterが必須です。今のログを見る限り単一インスタンスで動いていそうですが、水平スケール時はご留意を。
紙送り/カットの微調整：カット位置が気になる場合は .env で FEED_LINES を 1〜3 の範囲で調整、EPSON機で合わない時は CUT_MODE=B を試す。
*/

// gateway-print-client.js — RAW(9100/TCP)直送版：PNG(base64)→2値化→GS v 0
'use strict';
require('dotenv').config();

process.on('uncaughtException', function (err) {
  console.error('[FATAL] uncaughtException:', err && err.stack ? err.stack : err);
});

process.on('unhandledRejection', function (reason, p) {
  console.error('[FATAL] unhandledRejection:', reason);
});

const { io } = require('socket.io-client');
const sharp = require('sharp');
const net = require('net');

// ========= 環境変数 =========
const RAW_PRINTER_IP = process.env.PRINTER_IP || '127.0.0.1';
const PRINTER_IP = RAW_PRINTER_IP.trim();
const PRINTER_PORT = Number(process.env.PRINTER_PORT || 9100);
const SERVER_URL = process.env.SERVER_URL || '';
const GATEWAY_KEY = process.env.GATEWAY_KEY || '';

const RASTER_WIDTH = Number(process.env.RASTER_WIDTH || 384);
const THRESHOLD = Number(process.env.THRESHOLD || 128);
const FEED_LINES = Number(process.env.FEED_LINES || 2);
const CUT_MODE = (process.env.CUT_MODE || 'A').toUpperCase(); // 'A' or 'B'
const DEBUG = /^(1|true|yes)$/i.test(String(process.env.DEBUG || 'false'));
const RAW_PRINTER_ROLE = process.env.PRINTER_ROLE || 'any'; // 'receipt' | 'qr' | 'kitchen' | 'any'
const PRINTER_ROLE = RAW_PRINTER_ROLE.trim().toLowerCase();

// Socket.IO keepalive 調整（ネットワーク機器のアイドルタイムアウト対策）
const SIO_PING_INTERVAL = Number(process.env.SIO_PING_INTERVAL || 25000);
const SIO_PING_TIMEOUT = Number(process.env.SIO_PING_TIMEOUT || 60000);
const SIO_TIMEOUT = Number(process.env.SIO_TIMEOUT || 20000);

// 心拍（任意：サーバ側が TTL で可視化する想定）
const HEARTBEAT_INTERVAL_MS = Number(process.env.HEARTBEAT_INTERVAL_MS || 8000);
const USE_HTTP_HEARTBEAT = String(process.env.USE_HTTP_HEARTBEAT || 'false').toLowerCase() === 'true';

// プリンタ I/O
const PRINTER_IO_TIMEOUT_MS = Number(process.env.PRINTER_IO_TIMEOUT_MS || 7000);
const PRINT_MAX_RETRY = Number(process.env.PRINT_MAX_RETRY || 2);

function dlog() {
  if (DEBUG) {
    const args = Array.prototype.slice.call(arguments);
    args.unshift('[debug]');
    console.log.apply(console, args);
  }
}

// ---- fetch フォールバック（HTTP心拍で必要な場合のみ） ----
let _fetch = global.fetch;
async function safeFetch(url, opt) {
  if (!_fetch) {
    try {
      const mod = await import('node-fetch'); // npm i node-fetch
      _fetch = mod.default;
    } catch (_) { return null; }
  }
  try { return await _fetch(url, opt); } catch (_) { return null; }
}

console.log('[env] PRINTER_RAW', JSON.stringify(RAW_PRINTER_IP));
console.log('[env] PRINTER', PRINTER_IP + ':' + PRINTER_PORT);
console.log('[env] SERVER_URL', SERVER_URL);
console.log('[env] RASTER_WIDTH', RASTER_WIDTH, 'THRESHOLD', THRESHOLD, 'FEED_LINES', FEED_LINES, 'CUT', CUT_MODE);
console.log('[env] HEARTBEAT_INTERVAL_MS', HEARTBEAT_INTERVAL_MS, 'HTTP_HB', USE_HTTP_HEARTBEAT, 'DEBUG', DEBUG);
console.log('[env] PRINTER_ROLE', PRINTER_ROLE);

// ---- PNG(base64) → 1bpp packed（横:8pxごと1byte） ----
async function pngBase64ToPacked(pngBase64) {
  if (!pngBase64 || typeof pngBase64 !== 'string') {
    throw new Error('pngBase64 missing/invalid');
  }
  const b64 = pngBase64.replace(/^data:image\/\w+;base64,/, '');
  const buf = Buffer.from(b64, 'base64');
  dlog('png b64 len =', b64.length, 'buf bytes =', buf.length);

  // RGBA raw → リサイズ（幅 RASTER_WIDTH）→ 2値化
  const pipeline = sharp(buf)
    .resize({ width: RASTER_WIDTH, withoutEnlargement: true })
    .ensureAlpha()
    .threshold(THRESHOLD)
    .raw();

  const out = await pipeline.toBuffer({ resolveWithObject: true });
  const data = out.data;
  const info = out.info;
  const width = info.width;
  const height = info.height;

  // 1bit pack（白=0, 黒=1）
  const bytesPerRow = Math.ceil(width / 8);
  const packed = Buffer.alloc(bytesPerRow * height);

  for (let y = 0; y < height; y++) {
    for (let x = 0; x < width; x++) {
      const i = (y * width + x) * 4;
      const r = data[i];
      const a = data[i + 3];
      const isBlack = (a > 0 && r < 128); // threshold 後は r=g=b
      if (isBlack) {
        const byteIndex = y * bytesPerRow + (x >> 3);
        const bit = 7 - (x & 7);
        packed[byteIndex] |= (1 << bit);
      }
    }
  }

  return { packed, widthBytes: bytesPerRow, height, width };
}

// ---- RAW印字（GS v 0：raster bit image） ----
function printRasterRaw(payload, copies) {
  const widthBytes = payload.widthBytes;
  const height = payload.height;
  const packed = payload.packed;
  const copiesNum = copies || 1;

  return new Promise(function (resolve, reject) {
    const sock = net.createConnection({ host: PRINTER_IP, port: PRINTER_PORT }, function () {
      dlog('[raw] connected');

      // 初期化 ESC @
      sock.write(Buffer.from([0x1B, 0x40]));
      // 左寄せ ESC a 0
      sock.write(Buffer.from([0x1B, 0x61, 0x00]));

      for (let c = 0; c < copiesNum; c++) {
        // GS v 0 m=0（通常密度）
        const xL = widthBytes & 0xFF;
        const xH = (widthBytes >> 8) & 0xFF;
        const yL = height & 0xFF;
        const yH = (height >> 8) & 0xFF;

        sock.write(Buffer.from([0x1D, 0x76, 0x30, 0x00, xL, xH, yL, yH]));
        sock.write(packed);

        // 用紙送り ESC d n
        sock.write(Buffer.from([0x1B, 0x64, FEED_LINES & 0xFF]));
      }

      // カット
      // 機種差があるため A/B を切替可能に（A=0x41系, B=0x00系）
      const cutParam = (CUT_MODE === 'B') ? 0x00 : 0x41;
      sock.write(Buffer.from([0x1D, 0x56, cutParam, 0x10]));

      sock.end();
    });

    sock.setTimeout(PRINTER_IO_TIMEOUT_MS);
    sock.on('timeout', function () {
      console.error('[raw] timeout');
      try { sock.destroy(); } catch (_) { }
      reject(new Error('timeout'));
    });
    sock.on('error', function (e) {
      console.error('[raw] error', e && e.message ? e.message : e);
      reject(e);
    });
    sock.on('close', function () {
      dlog('[raw] closed');
      resolve();
    });
  });
}

// ========= ソケット接続（/gateway 名前空間） =========
const socket = io((SERVER_URL || '') + '/gateway', {
  transports: ['websocket'],
  auth: { gatewayKey: GATEWAY_KEY },
  reconnection: true,
  reconnectionAttempts: Infinity,
  reconnectionDelay: 1000,
  reconnectionDelayMax: 5000,
  timeout: SIO_TIMEOUT,
  pingInterval: SIO_PING_INTERVAL,
  pingTimeout: SIO_PING_TIMEOUT
});

let hbTimer = null;

function startHeartbeat() {
  stopHeartbeat();
  try { socket.emit('heartbeat'); } catch (_) { }

  hbTimer = setInterval(async function () {
    try { socket.emit('heartbeat'); } catch (_) { }
    if (USE_HTTP_HEARTBEAT && SERVER_URL) {
      const url = SERVER_URL.replace(/\/+$/, '') + '/staff/gateway/heartbeat';
      await safeFetch(url, { method: 'POST' });
    }
  }, HEARTBEAT_INTERVAL_MS);
}
function stopHeartbeat() {
  if (hbTimer) { clearInterval(hbTimer); hbTimer = null; }
}

socket.on('connect', function () {
  console.log('[gateway] connected:', socket.id);
  startHeartbeat();
});

socket.on('disconnect', function (reason) {
  // 期待通りの終了は静かに、想定外は警告に
  if (reason === 'transport close' || reason === 'io client disconnect') {
    dlog('[gateway] disconnected:', reason);
  } else {
    console.warn('[gateway] disconnected:', reason);
  }
  stopHeartbeat();
});

socket.on('connect_error', function (e) {
  console.warn('[gateway] connect_error:', e && e.message ? e.message : e);
  stopHeartbeat();
});

socket.on('reconnect_attempt', function (n) {
  dlog('[gateway] reconnect_attempt', n);
});

socket.on('reconnect_failed', function () {
  console.warn('[gateway] reconnect_failed (no more attempts?)');
});

socket.on('error', function (err) {
  console.error('[gateway] socket error', err && err.message ? err.message : err);
});

// --- 直近ジョブの重複抑止 ---
const RECENT_JOBS = new Map(); // jobId -> expiresAt(ms)
const DEDUP_TTL = Number(process.env.GATEWAY_DEDUP_TTL_MS || 4000);
function seenJob(jobId) {
  const now = Date.now();
  const exp = RECENT_JOBS.get(jobId) || 0;
  if (exp > now) return true;
  RECENT_JOBS.set(jobId, now + DEDUP_TTL);
  return false;
}
setInterval(() => {
  const now = Date.now();
  for (const [k, v] of RECENT_JOBS) if (v <= now) RECENT_JOBS.delete(k);
}, 10000).unref();

// 受信ハンドラ：{ jobId, pngBase64, copies, meta }
socket.on('printPng', async function (payload) {
  payload = payload || {};
  const jobId = payload.jobId;
  const pngBase64 = payload.pngBase64;
  const copies = payload.copies || 1;
  const meta = payload.meta || {};
  const src = meta.source || {};
  console.log('[gateway] receive job', jobId, '(copies=' + copies + ')', {
    kind: src.kind, id: src.id, b64len: pngBase64 ? pngBase64.length : 0
  });

  // === プリンターロールフィルタ ===
  // meta.source.printerRole が無いジョブは 'receipt' 扱いにする（既存運用との互換用）
  var jobRole = (src.printerRole || '').trim().toLowerCase();
  if (!jobRole) {
    switch (src.kind) {
      case 'qr-ticket':
        jobRole = 'qr';
        break;
      case 'order-ticket':
        jobRole = 'kitchen';
        break;
      default:
        jobRole = 'receipt';
    }
  }
  if (PRINTER_ROLE !== 'any' && jobRole !== PRINTER_ROLE) {
    console.log('[gateway] skip job for other role', {
      printerRoleEnv: PRINTER_ROLE,
      jobRole: jobRole,
      jobId: jobId
    });
    return;
  }

  if (!jobId) {
    console.warn('[gateway] missing jobId, proceed but cannot dedupe');
  } else if (seenJob(jobId)) {
    console.warn('[gateway] duplicate job suppressed', jobId);
    return;
  }

  try {
    if (!pngBase64 || typeof pngBase64 !== 'string' || pngBase64.length < 32) {
      throw new Error('missing/too short pngBase64');
    }

    const packedObj = await pngBase64ToPacked(pngBase64);
    dlog('[pack] size:', packedObj.width, 'x', packedObj.height,
      'bytes/row=', packedObj.widthBytes, 'total=', packedObj.packed.length);

    for (let attempt = 0; attempt <= PRINT_MAX_RETRY; attempt++) {
      try {
        await printRasterRaw(packedObj, copies);
        console.log('[gateway] printed job', jobId);
        return;
      } catch (e) {
        console.error('[gateway] print attempt ' + (attempt + 1) + ' failed:', e && e.message ? e.message : e);
        if (attempt === PRINT_MAX_RETRY) throw e;
        await new Promise(function (r) { setTimeout(r, 500); });
      }
    }
  } catch (e) {
    console.error('[gateway] print error', e);
  }
});
