/**
 * gateway-print-client.js — RAW(9100/TCP)直送版：PNG(base64)→2値化→GS v 0（帯送信 + 直列キュー）
 * =================================================================================
 * 重要な考え方：
 *  - RAW(9100)は「印刷完了」を返してくれないため、"timeout" を根拠に再送すると二重印刷になりやすい
 *  - よって、ジョブは必ず直列化（同時に2本プリンタへ送らない）
 *  - 送信完了後の timeout は「未知(UNKNOWN)」として扱い、基本は再送しない（= 二重印刷を防ぐ）
  */

'use strict';
require('dotenv').config();

const fs = require('fs');
const path = require('path');

process.on('uncaughtException', function (err) {
  console.error('[FATAL] uncaughtException:', err && err.stack ? err.stack : err);
});

process.on('unhandledRejection', function (reason) {
  console.error('[FATAL] unhandledRejection:', reason);
});

const { io } = require('socket.io-client');
const sharp = require('sharp');
const net = require('net');

// ========= 環境変数 =========
const RAW_PRINTER_IP = process.env.PRINTER_IP || '127.0.0.1';
const PRINTER_IP = String(RAW_PRINTER_IP).trim();
const PRINTER_PORT = Number(process.env.PRINTER_PORT || 9100);

const SERVER_URL = process.env.SERVER_URL || '';
const GATEWAY_KEY = process.env.GATEWAY_KEY || '';

const RASTER_WIDTH = Number(process.env.RASTER_WIDTH || 384);
const THRESHOLD = Number(process.env.THRESHOLD || 128);
const FEED_LINES = Number(process.env.FEED_LINES || 2);
const CUT_MODE = String(process.env.CUT_MODE || 'A').toUpperCase(); // 'A' or 'B'
const DEBUG = /^(1|true|yes)$/i.test(String(process.env.DEBUG || 'false'));

const RAW_PRINTER_ROLE = process.env.PRINTER_ROLE || 'any'; // 'receipt' | 'qr' | 'kitchen' | 'any'
const PRINTER_ROLE = String(RAW_PRINTER_ROLE).trim().toLowerCase();

// ============================
// スプール（未印刷をディスクに保持して復帰後に自動再印刷）
// ============================
const SPOOL_DIR = process.env.SPOOL_DIR
  ? String(process.env.SPOOL_DIR)
  : path.join(__dirname, 'spool-' + PRINTER_ROLE);
const SPOOL_PENDING_DIR = path.join(SPOOL_DIR, 'pending');
const SPOOL_DONE_DIR = path.join(SPOOL_DIR, 'done');

const SPOOL_MAX_AGE_MS = Number(process.env.SPOOL_MAX_AGE_MS || 24 * 60 * 60 * 1000); // 24h
const JOB_RETRY_BASE_MS = Number(process.env.JOB_RETRY_BASE_MS || 5000);               // 5s
const JOB_RETRY_MAX_MS = Number(process.env.JOB_RETRY_MAX_MS || 60000);                // 60s
const JOB_MAX_ATTEMPTS = Number(process.env.JOB_MAX_ATTEMPTS || 200);

function mkdirp(p) { try { fs.mkdirSync(p, { recursive: true }); } catch (_) { } }
mkdirp(SPOOL_PENDING_DIR);
mkdirp(SPOOL_DONE_DIR);

function safeId(id) {
  return String(id).replace(/[^a-zA-Z0-9._-]/g, '_');
}
function pendingPath(jobId) { return path.join(SPOOL_PENDING_DIR, safeId(jobId) + '.json'); }
function donePath(jobId) { return path.join(SPOOL_DONE_DIR, safeId(jobId) + '.json'); }

function writeJsonAtomic(filePath, obj) {
  const tmp = filePath + '.tmp';
  fs.writeFileSync(tmp, JSON.stringify(obj), 'utf8');
  fs.renameSync(tmp, filePath);
}

function readJsonSafe(filePath) {
  try { return JSON.parse(fs.readFileSync(filePath, 'utf8')); } catch (_) { return null; }
}

function spoolSavePending(job) {
  if (!job || !job.jobId) return false;
  const p = pendingPath(job.jobId);
  if (fs.existsSync(donePath(job.jobId))) return false;    // 既に完了扱い
  if (fs.existsSync(p)) return true;                       // 既にpendingあり（重複受信）
  const now = Date.now();
  const rec = {
    jobId: job.jobId,
    pngBase64: job.pngBase64,
    copies: job.copies || 1,
    src: job.src || {},
    jobRole: job.jobRole || 'receipt',
    createdAt: now,
    attempts: 0,
    nextRetryAt: 0
  };
  try { writeJsonAtomic(p, rec); return true; } catch (e) { console.error('[spool] save pending failed', e); return false; }
}

function spoolUpdateRetry(jobId, attempts, nextRetryAt) {
  const p = pendingPath(jobId);
  const rec = readJsonSafe(p);
  if (!rec) return;
  rec.attempts = attempts;
  rec.nextRetryAt = nextRetryAt;
  try { writeJsonAtomic(p, rec); } catch (e) { console.error('[spool] update retry failed', e); }
}

function spoolMarkDone(jobId, status) {
  if (!jobId) return;
  const p = pendingPath(jobId);
  const now = Date.now();
  let rec = null;
  if (fs.existsSync(p)) rec = readJsonSafe(p);
  if (!rec) rec = { jobId: jobId };
  rec.doneAt = now;
  rec.doneStatus = status || 'ok';
  // done は重複抑止用なので画像本体は保持しない（ディスク節約）
  if (rec.pngBase64) delete rec.pngBase64;

  const d = donePath(jobId);
  try {
    // pending があれば doneへ移動（内容保持）
    if (fs.existsSync(p)) {
      writeJsonAtomic(d, rec);
      try { fs.unlinkSync(p); } catch (_) { }
    } else {
      // pending無しでも done を作って重複抑止
      if (!fs.existsSync(d)) writeJsonAtomic(d, rec);
    }
  } catch (e) {
    console.error('[spool] mark done failed', e);
  }
}

function spoolCleanup() {
  const now = Date.now();
  // done の掃除（DEDUP_TTL を目安に消してOK）
  try {
    const files = fs.readdirSync(SPOOL_DONE_DIR);
    for (let i = 0; i < files.length; i++) {
      const f = files[i];
      if (!/\.json$/i.test(f)) continue;
      const full = path.join(SPOOL_DONE_DIR, f);
      const rec = readJsonSafe(full);
      const doneAt = rec && rec.doneAt ? Number(rec.doneAt) : 0;
      if (doneAt && (doneAt + DEDUP_TTL) < now) {
        try { fs.unlinkSync(full); } catch (_) { }
      }
    }
  } catch (_) { }

  // pending の期限切れ掃除
  try {
    const files = fs.readdirSync(SPOOL_PENDING_DIR);
    for (let i = 0; i < files.length; i++) {
      const f = files[i];
      if (!/\.json$/i.test(f)) continue;
      const full = path.join(SPOOL_PENDING_DIR, f);
      const rec = readJsonSafe(full);
      const createdAt = rec && rec.createdAt ? Number(rec.createdAt) : 0;
      if (createdAt && (createdAt + SPOOL_MAX_AGE_MS) < now) {
        console.warn('[spool] pending too old, drop', rec && rec.jobId ? rec.jobId : f);
        try { fs.unlinkSync(full); } catch (_) { }
      }
    }
  } catch (_) { }
}
setInterval(spoolCleanup, 60000).unref();


// Socket.IO keepalive
const SIO_PING_INTERVAL = Number(process.env.SIO_PING_INTERVAL || 25000);
const SIO_PING_TIMEOUT = Number(process.env.SIO_PING_TIMEOUT || 60000);
const SIO_TIMEOUT = Number(process.env.SIO_TIMEOUT || 20000);

// 心拍（任意）
const HEARTBEAT_INTERVAL_MS = Number(process.env.HEARTBEAT_INTERVAL_MS || 8000);
const USE_HTTP_HEARTBEAT = String(process.env.USE_HTTP_HEARTBEAT || 'false').toLowerCase() === 'true';

// プリンタI/O
const PRINTER_IO_TIMEOUT_MS = Number(process.env.PRINTER_IO_TIMEOUT_MS || 90000);
const PRINT_MAX_RETRY = Number(process.env.PRINT_MAX_RETRY || 0); // ★二重印刷防止のため基本0推奨

// 帯送信
const RASTER_BAND_HEIGHT = Number(process.env.RASTER_BAND_HEIGHT || 256);
const RASTER_BAND_DELAY_MS = Number(process.env.RASTER_BAND_DELAY_MS || 8);

// リトライ待ち / クールダウン
const PRINT_RETRY_WAIT_MS = Number(process.env.PRINT_RETRY_WAIT_MS || 8000);
const PRINT_COOLDOWN_MS = Number(process.env.PRINT_COOLDOWN_MS || 15000);

// soft reset
const AUTO_RESET_ON_FAIL = String(process.env.AUTO_RESET_ON_FAIL || 'false').toLowerCase() === 'true';
const RESET_CONNECT_TIMEOUT_MS = Number(process.env.RESET_CONNECT_TIMEOUT_MS || 2000);

// DEDUP（★数分後の再送にも効かせる）
const DEDUP_TTL = Number(process.env.GATEWAY_DEDUP_TTL_MS || 600000);

function dlog() {
  if (DEBUG) {
    const args = Array.prototype.slice.call(arguments);
    args.unshift('[debug]');
    console.log.apply(console, args);
  }
}

function sleep(ms) {
  return new Promise(function (r) { setTimeout(r, ms); });
}

// ---- fetch フォールバック（HTTP心拍で必要な場合のみ） ----
let _fetch = global.fetch;
async function safeFetch(url, opt) {
  if (!_fetch) {
    try {
      const mod = await import('node-fetch');
      _fetch = mod.default;
    } catch (_) { return null; }
  }
  try { return await _fetch(url, opt); } catch (_) { return null; }
}

console.log('[env] PRINTER_RAW', JSON.stringify(RAW_PRINTER_IP));
console.log('[env] PRINTER', PRINTER_IP + ':' + PRINTER_PORT);
console.log('[env] SERVER_URL', SERVER_URL);
console.log('[env] RASTER_WIDTH', RASTER_WIDTH, 'THRESHOLD', THRESHOLD, 'FEED_LINES', FEED_LINES, 'CUT', CUT_MODE);
console.log('[env] HEARTBEAT_INTERVAL_MS', HEARTBEAT_INTERVAL_MS, 'HTTP_HB', USE_HTTP_HEARTBEAT, 'DEBUG', DEBUG);
console.log('[env] PRINTER_ROLE', PRINTER_ROLE);
console.log('[env] PRINTER_IO_TIMEOUT_MS', PRINTER_IO_TIMEOUT_MS,
  'BAND_H', RASTER_BAND_HEIGHT, 'BAND_DELAY_MS', RASTER_BAND_DELAY_MS,
  'RETRY_WAIT_MS', PRINT_RETRY_WAIT_MS, 'COOLDOWN_MS', PRINT_COOLDOWN_MS,
  'AUTO_RESET_ON_FAIL', AUTO_RESET_ON_FAIL,
  'DEDUP_TTL', DEDUP_TTL,
  'PRINT_MAX_RETRY', PRINT_MAX_RETRY
);

// ============================
// PNG(base64) → 1bpp packed
// ============================
async function pngBase64ToPacked(pngBase64) {
  if (!pngBase64 || typeof pngBase64 !== 'string') {
    throw new Error('pngBase64 missing/invalid');
  }
  const b64 = pngBase64.replace(/^data:image\/\w+;base64,/, '');
  const buf = Buffer.from(b64, 'base64');
  dlog('png b64 len =', b64.length, 'buf bytes =', buf.length);

  const pipeline = sharp(buf)
    .resize({ width: RASTER_WIDTH, withoutEnlargement: true })
    .ensureAlpha()
    .threshold(THRESHOLD)
    .raw();

  const out = await pipeline.toBuffer({ resolveWithObject: true });
  const data = out.data;
  const info = out.info;
  const width = info.width;
  const height = info.height;

  const bytesPerRow = Math.ceil(width / 8);
  const packed = Buffer.alloc(bytesPerRow * height);

  for (let y = 0; y < height; y++) {
    for (let x = 0; x < width; x++) {
      const i = (y * width + x) * 4;
      const r = data[i];
      const a = data[i + 3];
      const isBlack = (a > 0 && r < 128);
      if (isBlack) {
        const byteIndex = y * bytesPerRow + (x >> 3);
        const bit = 7 - (x & 7);
        packed[byteIndex] |= (1 << bit);
      }
    }
  }

  return { packed, widthBytes: bytesPerRow, height, width };
}

// ============================
// RAW印字（帯送信 + drain待ち）
// ============================
function writeAll(sock, buf) {
  return new Promise(function (resolve, reject) {
    try {
      const ok = sock.write(buf);
      if (ok) return resolve();
      sock.once('drain', resolve);
    } catch (e) {
      reject(e);
    }
  });
}

function makeErr(code, msg) {
  const e = new Error(msg || code);
  e.code = code;
  return e;
}

/**
 * soft reset（ESC @）を軽く投げる
 * ※ AUTO_RESET_ON_FAIL=false 推奨（処理中のプリンタに刺激を与えることもあるため）
 */
function trySoftResetPrinter() {
  return new Promise(function (resolve) {
    const sock = net.createConnection({ host: PRINTER_IP, port: PRINTER_PORT }, function () {
      try { sock.write(Buffer.from([0x1B, 0x40])); } catch (_) { }
      setTimeout(function () {
        try { sock.end(); } catch (_) { }
        resolve(true);
      }, 200);
    });

    sock.setTimeout(RESET_CONNECT_TIMEOUT_MS);
    sock.on('timeout', function () { try { sock.destroy(); } catch (_) { } resolve(false); });
    sock.on('error', function () { try { sock.destroy(); } catch (_) { } resolve(false); });
  });
}

/**
 * printRasterRaw
 * 戻り値：
 *  - { status:'ok' }         … closeまで素直に完了
 *  - { status:'unknown' }    … 送信は終えたが、その後 timeout（＝実際に印字された可能性あり）
 *
 * ★ポイント：
 *  - "unknown" を「失敗」として再送しない（＝二重印刷を避ける）
 */
function printRasterRaw(payload, copies) {
  const widthBytes = payload.widthBytes;
  const height = payload.height;
  const packed = payload.packed;
  const copiesNum = copies || 1;

  const BAND = Math.max(16, Math.min(2048, RASTER_BAND_HEIGHT));

  return new Promise(function (resolve, reject) {
    let sentAll = false; // ★ここが重要：送信完了後のtimeoutはUNKNOWN扱いにする
    let settled = false;
    function ok(v) { if (settled) return; settled = true; resolve(v); }
    function ng(e) { if (settled) return; settled = true; reject(e); }
    const sock = net.createConnection({ host: PRINTER_IP, port: PRINTER_PORT }, function () {
      (async function () {
        try {
          dlog('[raw] connected');

          await writeAll(sock, Buffer.from([0x1B, 0x40]));       // ESC @
          await writeAll(sock, Buffer.from([0x1B, 0x61, 0x00])); // ESC a 0

          for (let c = 0; c < copiesNum; c++) {
            for (let y = 0; y < height; y += BAND) {
              const bandH = Math.min(BAND, height - y);

              const xL = widthBytes & 0xFF;
              const xH = (widthBytes >> 8) & 0xFF;
              const yL = bandH & 0xFF;
              const yH = (bandH >> 8) & 0xFF;

              const start = y * widthBytes;
              const end = start + bandH * widthBytes;
              const slice = packed.slice(start, end);

              await writeAll(sock, Buffer.from([0x1D, 0x76, 0x30, 0x00, xL, xH, yL, yH]));
              await writeAll(sock, slice);

              if (RASTER_BAND_DELAY_MS > 0) await sleep(RASTER_BAND_DELAY_MS);
            }

            await writeAll(sock, Buffer.from([0x1B, 0x64, FEED_LINES & 0xFF])); // ESC d
          }

          const cutParam = (CUT_MODE === 'B') ? 0x00 : 0x41;
          await writeAll(sock, Buffer.from([0x1D, 0x56, cutParam, 0x10])); // GS V

          // ★ここまで来たら「送信は完了」
          sentAll = true;

          sock.end();
        } catch (e) {
          try { sock.destroy(); } catch (_) { }
          ng(e);
        }
      })();
    });

    sock.setTimeout(PRINTER_IO_TIMEOUT_MS);

    sock.on('timeout', function () {
      console.error('[raw] timeout (sentAll=' + sentAll + ')');
      try { sock.destroy(); } catch (_) { }

      // ★送信完了後のtimeoutは「unknown」扱い（再送しない）
      if (sentAll) return ok({ status: 'unknown' });
      return ng(makeErr('ETIMEOUT_BEFORE_SEND', 'timeout(before send complete)'));
    });

    sock.on('error', function (e) {
      console.error('[raw] error', e && e.message ? e.message : e);

      // ★送信完了後に落ちたなら unknown 扱いに寄せる（再送による二重印刷を避ける）
      if (sentAll) return ok({ status: 'unknown' });
      // 送信前/途中で落ちた → 後で再試行できる可能性
      if (e && !e.code) e.code = 'EPRINT_SOCKET';
      ng(e);
    });

    sock.on('close', function () {
      dlog('[raw] closed');
      ok({ status: 'ok' });
    });
  });
}

// ============================
// Socket.IO
// ============================
const socket = io((SERVER_URL || '') + '/gateway', {
  transports: ['websocket'],
  auth: { gatewayKey: GATEWAY_KEY, printerRole: PRINTER_ROLE },
  reconnection: true,
  reconnectionAttempts: Infinity,
  reconnectionDelay: 1000,
  reconnectionDelayMax: 5000,
  timeout: SIO_TIMEOUT,
  pingInterval: SIO_PING_INTERVAL,
  pingTimeout: SIO_PING_TIMEOUT
});

let hbTimer = null;
function startHeartbeat() {
  stopHeartbeat();
  try { socket.emit('heartbeat', { role: PRINTER_ROLE }); } catch (_) { }

  hbTimer = setInterval(async function () {
    try { socket.emit('heartbeat', { role: PRINTER_ROLE }); } catch (_) { }
    if (USE_HTTP_HEARTBEAT && SERVER_URL) {
      const url = SERVER_URL.replace(/\/+$/, '') + '/staff/gateway/heartbeat';
      await safeFetch(url, {
        method: 'POST',
        headers: { 'content-type': 'application/json' },
        body: JSON.stringify({ role: PRINTER_ROLE })
      });
    }
  }, HEARTBEAT_INTERVAL_MS);
}
function stopHeartbeat() {
  if (hbTimer) { clearInterval(hbTimer); hbTimer = null; }
}

socket.on('connect', function () {
  console.log('[gateway] connected:', socket.id);
  startHeartbeat();
});
socket.on('disconnect', function (reason) {
  if (reason === 'transport close' || reason === 'io client disconnect') {
    dlog('[gateway] disconnected:', reason);
  } else {
    console.warn('[gateway] disconnected:', reason);
  }
  stopHeartbeat();
});
socket.on('connect_error', function (e) {
  console.warn('[gateway] connect_error:', e && e.message ? e.message : e);
  stopHeartbeat();
});
socket.on('reconnect_attempt', function (n) { dlog('[gateway] reconnect_attempt', n); });
socket.on('reconnect_failed', function () { console.warn('[gateway] reconnect_failed'); });
socket.on('error', function (err) {
  console.error('[gateway] socket error', err && err.message ? err.message : err);
});

// ============================
// 直近ジョブの重複抑止
// ============================
const RECENT_JOBS = new Map(); // jobId -> expiresAt(ms)
function seenJob(jobId) {
  const now = Date.now();
  const exp = RECENT_JOBS.get(jobId) || 0;
  if (exp > now) return true;

  // ★再起動後も重複を抑える：done に残っていたら「見た」扱い
  try {
    if (fs.existsSync(donePath(jobId))) {
      // doneAt + DEDUP_TTL の範囲で抑止
      const rec = readJsonSafe(donePath(jobId));
      const doneAt = rec && rec.doneAt ? Number(rec.doneAt) : now;
      const exp2 = doneAt + DEDUP_TTL;
      if (exp2 > now) {
        RECENT_JOBS.set(jobId, exp2);
        return true;
      }
    }
  } catch (_) { }
  RECENT_JOBS.set(jobId, now + DEDUP_TTL);
  return false;
}
setInterval(function () {
  const now = Date.now();
  for (const [k, v] of RECENT_JOBS) {
    if (v <= now) RECENT_JOBS.delete(k);
  }
}, 30000).unref();

// ============================
// 印刷ジョブ直列キュー（ここが二重印刷対策の要）
// ============================
const JOB_QUEUE = [];
let printing = false;
let coolDownUntil = 0;
const INFLIGHT = new Set(); // jobId（同一ジョブの多重enqueue防止）

function enqueueJob(job) {
  if (job && job.jobId) {
    if (INFLIGHT.has(job.jobId)) return;
    INFLIGHT.add(job.jobId);
  }
  JOB_QUEUE.push(job);
  pumpQueue();
}

async function pumpQueue() {
  if (printing) return;
  printing = true;

  while (JOB_QUEUE.length) {
    const job = JOB_QUEUE.shift();
    await handleOneJob(job);
  }

  printing = false;
}

async function handleOneJob(job) {
  const jobId = job.jobId;
  const pngBase64 = job.pngBase64;
  const copies = job.copies || 1;
  const src = job.src || {};
  const jobRole = job.jobRole || 'receipt';

  // cooldown（プリンタを追い込まない）
  const now = Date.now();
  if (coolDownUntil > now) {
    const waitMs = coolDownUntil - now;
    console.warn('[gateway] cooldown active, wait', waitMs, 'ms before printing job', jobId);
    await sleep(waitMs);
  }

  // 印刷実行
  try {
    const packedObj = await pngBase64ToPacked(pngBase64);
    dlog('[pack] size:', packedObj.width, 'x', packedObj.height,
      'bytes/row=', packedObj.widthBytes, 'total=', packedObj.packed.length);

    for (let attempt = 0; attempt <= PRINT_MAX_RETRY; attempt++) {
      try {
        const res = await printRasterRaw(packedObj, copies);

        if (res && res.status === 'unknown') {
          // ★送信完了後timeout：再送しない（=二重印刷防止）
          console.warn('[gateway] print result UNKNOWN (likely printed later). do NOT retry. jobId=', jobId);
          spoolMarkDone(jobId, 'unknown');
          return;
        }

        console.log('[gateway] printed job', jobId);
        spoolMarkDone(jobId, 'ok');
        return;

      } catch (e) {
        console.error('[gateway] print attempt ' + (attempt + 1) + ' failed:', e && e.message ? e.message : e);

        // 失敗後クールダウン
        coolDownUntil = Date.now() + PRINT_COOLDOWN_MS;

        // soft reset（基本OFF推奨）
        if (AUTO_RESET_ON_FAIL) {
          const ok = await trySoftResetPrinter();
          console.warn('[gateway] soft reset tried:', ok ? 'OK' : 'NG');
        }

        if (attempt === PRINT_MAX_RETRY) throw e;
        await sleep(PRINT_RETRY_WAIT_MS);
      }
    }
  } catch (e) {
    console.error('[gateway] print error', e);
    coolDownUntil = Date.now() + PRINT_COOLDOWN_MS;

    // ★ここが本題：送信前/途中で落ちた系はスプールに残して後で自動再印刷
    if (jobId) {
      const code = String(e && e.code ? e.code : '');
      const retryable =
        code === 'ETIMEOUT_BEFORE_SEND' ||
        code === 'ECONNREFUSED' || code === 'EHOSTUNREACH' || code === 'ENETUNREACH' ||
        code === 'ETIMEDOUT' || code === 'EPIPE' || code === 'ECONNRESET' || code === 'EPRINT_SOCKET';

      // それっぽい code が無い場合も「プリンタ落ち」っぽければ救う
      const msg = String(e && e.message ? e.message : '');
      const looksLikeOffline = /ECONN|EHOST|ENET|timed out|socket|connect/i.test(msg);

      if (retryable || looksLikeOffline) {
        // スプールから attempts を読む
        const rec = readJsonSafe(pendingPath(jobId));
        const attempts = rec && rec.attempts ? Number(rec.attempts) : 0;
        const createdAt = rec && rec.createdAt ? Number(rec.createdAt) : Date.now();
        const age = Date.now() - createdAt;

        if (attempts >= JOB_MAX_ATTEMPTS || age > SPOOL_MAX_AGE_MS) {
          console.warn('[gateway] pending give up (too many attempts/too old). jobId=', jobId, 'attempts=', attempts, 'age(ms)=', age);
          // ここでは「done」にせず pending を残す運用もあり。今回は溜まり続け防止で done に逃がす案：
          spoolMarkDone(jobId, 'giveup');
          return;
        }

        const delay = Math.min(JOB_RETRY_MAX_MS, JOB_RETRY_BASE_MS * Math.pow(2, Math.min(10, attempts)));
        const nextRetryAt = Date.now() + delay;
        spoolUpdateRetry(jobId, attempts + 1, nextRetryAt);

        console.warn('[gateway] will retry later. jobId=', jobId, 'after(ms)=', delay, 'attempts=', attempts + 1);
        setTimeout(function () {
          const r = readJsonSafe(pendingPath(jobId));
          if (!r) return;
          // 期限前ならスキップ（scanで拾われる）
          if (r.nextRetryAt && Number(r.nextRetryAt) > Date.now()) return;
          enqueueJob({ jobId: r.jobId, pngBase64: r.pngBase64, copies: r.copies || 1, src: r.src || {}, jobRole: r.jobRole || 'receipt' });
        }, delay).unref();
        return;
      }
    }
  }
  finally {
    if (jobId) INFLIGHT.delete(jobId);
  }
}

// ============================
// 受信ハンドラ（ここでは印刷せず、キューに積むだけ）
// ============================
socket.on('printPng', function (payload) {
  payload = payload || {};

  const jobId = payload.jobId;
  const pngBase64 = payload.pngBase64;
  const copies = payload.copies || 1;
  const meta = payload.meta || {};
  const src = meta.source || {};

  console.log('[gateway] receive job', jobId, '(copies=' + copies + ')', {
    kind: src.kind, id: src.id, b64len: pngBase64 ? pngBase64.length : 0
  });

  // ロール判定（既存互換）
  let jobRole = String(src.printerRole || '').trim().toLowerCase();
  if (!jobRole) {
    switch (src.kind) {
      case 'qr-ticket': jobRole = 'qr'; break;
      case 'order-ticket': jobRole = 'kitchen'; break;
      default: jobRole = 'receipt';
    }
  }

  if (PRINTER_ROLE !== 'any' && jobRole !== PRINTER_ROLE) {
    console.log('[gateway] skip job for other role', {
      printerRoleEnv: PRINTER_ROLE,
      jobRole: jobRole,
      jobId: jobId
    });
    return;
  }

  if (!jobId) {
    console.warn('[gateway] missing jobId, proceed but cannot dedupe');
  } else if (seenJob(jobId)) {
    console.warn('[gateway] duplicate job suppressed', jobId);
    return;
  }

  if (!pngBase64 || typeof pngBase64 !== 'string' || pngBase64.length < 32) {
    console.error('[gateway] invalid pngBase64; skip. jobId=', jobId);
    return;
  }

  // ★ここがポイント：印刷は「必ずキューで直列化」
  const j = { jobId, pngBase64, copies, src, jobRole };
  // ★まずスプールに保存（落ちても復帰後に拾える）
  spoolSavePending(j);
  enqueueJob(j);
});

// 起動時：pending を拾って印刷再開
function loadPendingOnBoot() {
  try {
    const files = fs.readdirSync(SPOOL_PENDING_DIR);
    const recs = [];

    for (let i = 0; i < files.length; i++) {
      const f = files[i];
      if (!/\.json$/i.test(f)) continue;
      const full = path.join(SPOOL_PENDING_DIR, f);
      const rec = readJsonSafe(full);
      if (!rec || !rec.jobId || !rec.pngBase64) continue;
      if (rec.nextRetryAt && Number(rec.nextRetryAt) > Date.now()) continue;
      recs.push(rec);
    }
    recs.sort(function (a, b) { return (Number(a.createdAt) || 0) - (Number(b.createdAt) || 0); });
    for (let j = 0; j < recs.length; j++) {
      const rec = recs[j];
      enqueueJob({ jobId: rec.jobId, pngBase64: rec.pngBase64, copies: rec.copies || 1, src: rec.src || {}, jobRole: rec.jobRole || 'receipt' });
    }
  } catch (_) { }
}
loadPendingOnBoot();

// 定期的に pending をスキャンして「復帰後の取りこぼし」を拾う
setInterval(function () {
  if (printing) return;
  loadPendingOnBoot();
}, 3000).unref();
